import torch
import torch.nn as nn
import math
import torch.nn.functional as F
import torchvision

from einops import repeat
from einops.layers.torch import Rearrange


class OurModel(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size
        self.order=order

        self.encoder = nn.Linear(input_size, self.hidden_size, False)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)

        #adopt orthogonal initialization
        # nn.init.orthogonal(self.encoder.weight)
        # nn.init.orthogonal(self.decoder.weight)

    def forward(self, input):
        output = self.encoder(input)
        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output


class OurModel_VIT(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel_VIT, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size
        self.order=order

        # self.encoder = nn.Linear(input_size, self.hidden_size, False)
        # self.decoder = nn.Linear(order*self.hidden_size, output_size)
        self.decoder = nn.Linear(order*17*self.hidden_size, output_size)

        image_height, image_width = (32,32)
        patch_height, patch_width = (8,8)#(4,4)

        assert image_height % patch_height == 0 and image_width % patch_width == 0, 'Image dimensions must be divisible by the patch size.'

        num_patches = (image_height // patch_height) * (image_width // patch_width)
        patch_dim = 3 * patch_height * patch_width

        self.to_patch_embedding = nn.Sequential(
            Rearrange('b c (h p1) (w p2) -> b (h w) (p1 p2 c)', p1 = patch_height, p2 = patch_width),
            #nn.Linear(patch_dim, dim),
            nn.Linear(patch_dim, self.hidden_size, False)
        )

        self.pos_embedding = nn.Parameter(torch.randn(1, num_patches + 1, self.hidden_size))
        self.cls_token = nn.Parameter(torch.randn(1, 1, self.hidden_size))

        #adopt orthogonal initialization
        # nn.init.orthogonal(self.encoder.weight)
        # nn.init.orthogonal(self.decoder.weight)

    def forward(self, input):
        x = self.to_patch_embedding(input)
        b, n, _ = x.shape

        cls_tokens = repeat(self.cls_token, '() n d -> b n d', b = b)
        output = torch.cat((cls_tokens, x), dim=1)
        output += self.pos_embedding[:, :(n + 1)]

        output = output.reshape(b,-1)
        #output = self.encoder(input)
        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output

class OurModel_hybrid(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel_hybrid, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size
        self.order=order
        self.model = torchvision.models.resnet34(True).cuda()

        self.encoder = nn.Linear(input_size, self.hidden_size, False)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)

    def forward(self, input):
        input = self.model(input)
        output = self.encoder(input)
        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output

class OurModel_hybrid_512(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel_hybrid_512, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size
        self.order=order
        # self.model = nn.Sequential(*list(torchvision.models.resnet34(True).children())[:8]).cuda() #the 512 setting
        self.model = nn.Sequential(*list(torchvision.models.resnet34(True).children())[:5]).cuda()
        

        self.encoder = nn.Linear(input_size, self.hidden_size, False)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)

    def forward(self, input):
        input = self.model(input)
        print(input.shape)
        exit()
        input = input.reshape(input.shape[0],-1)
        output = self.encoder(input)
        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output

class OurModel_hybrid_pretrained_on_cifar(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel_hybrid_pretrained_on_cifar, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size
        self.order=order
        self.model = nn.Sequential(*list(torch.hub.load("chenyaofo/pytorch-cifar-models", "cifar10_resnet32", pretrained=True).children())[:7]).cuda()
        

        self.encoder = nn.Linear(input_size, self.hidden_size, False)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)

    def forward(self, input):
        input = self.model(input)
        input = input.reshape(input.shape[0],-1)
        output = self.encoder(input)
        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output


class OurModel_VIT_true(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel_VIT_true, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size
        self.order=order

        self.encoder = nn.Linear(input_size, self.hidden_size, False)
        # self.decoder = nn.Linear(order*self.hidden_size, output_size)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)

        image_height, image_width = (32,32)
        patch_height, patch_width = (16,16)#(16,16)#(8,8)#(4,4)
        dim = 256#256#256#64#16

        assert image_height % patch_height == 0 and image_width % patch_width == 0, 'Image dimensions must be divisible by the patch size.'

        num_patches = (image_height // patch_height) * (image_width // patch_width)
        patch_dim = 3 * patch_height * patch_width

        self.to_patch_embedding = nn.Sequential(
            Rearrange('b c (h p1) (w p2) -> b (h w) (p1 p2 c)', p1 = patch_height, p2 = patch_width),
            nn.Linear(patch_dim, dim),
            nn.ReLU()
            #nn.Linear(patch_dim, self.hidden_size, False)
        )

        #self.pos_embedding = nn.Parameter(torch.randn(1, num_patches + 1, dim))
        #self.cls_token = nn.Parameter(torch.randn(1, 1, dim))

        #adopt orthogonal initialization
        # nn.init.orthogonal(self.encoder.weight)
        # nn.init.orthogonal(self.decoder.weight)

    def forward(self, input):
        x = self.to_patch_embedding(input)
        b, n, _ = x.shape

        #cls_tokens = repeat(self.cls_token, '() n d -> b n d', b = b)
        #output = torch.cat((cls_tokens, x), dim=1)
        #output += self.pos_embedding[:, :(n + 1)]

        output = x.reshape(b,-1)
        # output = output.reshape(b,-1)
        output = self.encoder(output)

        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output

class OurModel_NN_encoding(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel_NN_encoding, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(1024+order+1, order)
        self.output_size = output_size
        self.order=order

        self.preencoder = nn.Linear(input_size, 1024)
        self.preencoder_2 = nn.Linear(1024, 512)

        self.encoder = nn.Linear(512, self.hidden_size, False)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)

        #adopt orthogonal initialization
        # nn.init.orthogonal(self.encoder.weight)
        # nn.init.orthogonal(self.decoder.weight)

    def forward(self, input):
        output =  F.relu(self.preencoder(input.reshape(input.shape[0],-1)))
        output =  F.relu(self.preencoder_2(output))
        output = self.encoder(output)

        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output

class OurModel_pooling(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(OurModel_pooling, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size
        self.order=order

        # self.pooling = nn.MaxPool2d(3,stride=2)
        self.pooling = nn.AvgPool2d(3,stride=2)
        self.encoder = nn.Linear(input_size, self.hidden_size, False)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)

        #adopt orthogonal initialization
        # nn.init.orthogonal(self.encoder.weight)
        # nn.init.orthogonal(self.decoder.weight)

    def forward(self, input):
        input = torch.mean(input,1)#mean on the channel dimension
        output = input.reshape(input.shape[0],-1)
        # output = self.pooling(input).reshape(input.shape[0],-1)

        output = self.encoder(output)
        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output

class OurModel_app(nn.Module):
    def __init__(self, input_size,  order,hidden_size, output_size=1):
        super(OurModel_app, self).__init__() 
        self.input_size = input_size
        self.hidden_size = hidden_size
        self.output_size = output_size
        self.order=order

        self.encoder = nn.Linear(input_size, self.hidden_size, False)
        self.decoder = nn.Linear(order*self.hidden_size, output_size)


    def forward(self, input):
        output = self.encoder(input)
        out=-torch.unsqueeze(output,dim=1)
        for i in range(1,self.order):
            out=torch.cat((out,torch.pow(-output,i+1).unsqueeze_(dim=1)),1)
        out=torch.flatten(out, 1)
        output = self.decoder(out)
        return output


class Feed_foward_same_depth(nn.Module):
    def __init__(self, input_size,  hidden_size, number_of_hidden_layer,output_size=1):
        super(Feed_foward_same_depth, self).__init__() 
        self.hidden_size = hidden_size
        self.layer_first = nn.Linear(input_size, self.hidden_size)
        self.layer = nn.ModuleList()  
        for i in range(number_of_hidden_layer):
            self.layer.append(nn.Linear(self.hidden_size, self.hidden_size))
            self.layer.append(nn.ReLU())
        self.layer_last = nn.Linear(self.hidden_size, output_size)

    def forward(self, input):
        out = F.relu(self.layer_first(input))
        for layer in self.layer:
            out = layer(out)
        out = self.layer_last(out)
        return out

class Feed_foward_same_width(nn.Module):
    def __init__(self, input_size,  order, output_size=1):
        super(Feed_foward_same_width, self).__init__() 
        self.input_size = input_size
        self.hidden_size = math.comb(input_size+order+1, order)
        self.output_size = output_size

        self.layer1 = nn.Linear(input_size, self.hidden_size)
        self.layer2 = nn.Linear(self.hidden_size, self.output_size)

    def forward(self, input):
        out = F.relu(self.layer1(input))
        out = self.layer2(out)
        return out

class ResNet(nn.Module):
    """
    ResNet model
    Arguments:
        block (BasicBlock or Bottleneck): type of basic block to be used.
        num_blocks (list): number of blocks in each sub-module.
        num_classes (int): number of output classes.
        device (torch.device or str): device to work on. 
    """
    def __init__(self, block, num_blocks, num_classes=10, device='cpu'):
        super(ResNet, self).__init__()
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512 * block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out

class BasicBlock(nn.Module):
    """
    Implements a basic block module for Resnets.
    Arguments:
        in_planes (int): number of input planes.
        out_planes (int): number of output filters.
        stride (int): stride of convolution.
    """
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out
def resnet():
    return ResNet(BasicBlock, [2, 2, 2, 2])